local lume = require "util.lume"
local playerutil = require "util.playerutil"
local soundutil = require "util.soundutil"
require "netcomponents"

local Brains = {}
local StateGraphs = {}
local Components = {}
local BossCoroutines = {}

local function LoadComponent(name)
	local cmp = Components[name]
	if cmp == nil then
		cmp = require("components."..name)
		assert(cmp ~= nil, "could not load component "..name)
		Components[name] = cmp
	end
	return cmp
end

local function LoadBrain(name)
	local brain = Brains[name]
	if brain == nil then
		brain = require("brains."..name)
		assert(brain ~= nil, "could not load brain "..name)
		Brains[name] = brain
	end
	return brain
end

local function LoadStateGraph(name, prefab)
	local key = name..":"..prefab
	local key_sg = StateGraphs[key]
	if key_sg == nil then
		local sg = StateGraphs[name]
		if sg == nil then
			sg = require("stategraphs."..name)
			assert(sg ~= nil, "could not load stategraph "..name)
			StateGraphs[name] = sg
		end
		key_sg = deepcopy(sg)
		StateGraphs[key] = key_sg
	end
	return key_sg
end

local function LoadBossCoro(name)
	local bc = BossCoroutines[name]
	if bc == nil then
		bc = require("bosscoroutines."..name)
		assert(bc ~= nil, "could not load boss coroutine "..name)
		BossCoroutines[name] = bc
	end
	return bc
end

local function UncacheStateGraph(name, prefab)
	local key = name..":"..prefab
	StateGraphs[key] = nil
	-- also force a reload on our base stategraph
	StateGraphs[name] = nil
	package.loaded["stategraphs."..name] = nil
end

local function GenerateBlankStateGraph(name)
	local events =
	{
	}

	local states =
	{
		State({
			name = "idle",
		}),
	}

	local sg = StateGraph(name, states, events, "idle")
	sg.autoGenerated = true
	return sg
end

function GenerateStateGraph(name)
	local sg = StateGraphs[name]
	if not sg then
		sg = GenerateBlankStateGraph(name)
		StateGraphs[name] = sg
	end
	return sg
end


EntityScript = Class(function(self, entity)
	self.entity = entity
	self.components = {}
	self.netcomponents = {}
	self.nrnetcomponents = 0	-- Ugh lua, why can't you just give me the nr of pairs in the table.
	self.GUID = entity:GetGUID()
	self.spawntime = GetTime()
	self.persists = true
	self.paused = false
	self._netinlimbo = false

	self.data = nil
	self.listeners = nil
	self.updatecomponents = nil
	self.newupdatecomponents = nil
	self.wallupdatecomponents = nil
	self.newwallupdatecomponents = nil
	self.brain = nil
	self.sg = nil
	self.eventlisteners = nil
	self.eventlistening = nil
	self.tasks = nil
end)

EntityScript.DebugNodeName = "DebugEntity"

-- Only call from SpawnPrefab.
function EntityScript:_SetSpawnInstigator(instigator)
	if self.SoundEmitter then
		local soundtracker = self.components.soundtracker or soundutil.FindSoundTracker(instigator)
		if soundtracker then
			soundtracker:ApplyUniquenessState(self)
		end
		-- else: probably spawned the world.
	end

	if self.OnSetSpawnInstigator then
		self:OnSetSpawnInstigator(instigator)
	end
end

function EntityScript:GetSaveRecord()
	local record =
	{
		data = self:GetPersistData(),
	}

	if self.Transform ~= nil then
		local x, y, z = self.Transform:GetWorldPosition()
		record.x = lume.round(x, 0.01)
		record.z = lume.round(z, 0.01)
		--y is often 0 in our game, so be selective.
		y = lume.round(y, 0.01)
		if y ~= 0 then
			record.y = y
		end

		--rot is often 0 in our game, so be selective.
		local rot = self.Transform:GetRotation()
		rot = lume.round(rot)
		if rot ~= 0 then
			record.rot = rot
		end
	end

	return record
end

function EntityScript:Hide()
	self.entity:Hide()
end

function EntityScript:Show()
	self.entity:Show()
end

--NOTE: this checks parents as well
function EntityScript:IsVisible()
	return self.entity:IsVisible()
end

function EntityScript:IsValid()
	return self.entity:IsValid()
end

function EntityScript:IsAsleep()
	return not self.entity:IsAwake()
end

function EntityScript:IsNetworked()
	return self.Network and self.Network:IsNetworked()
end

function EntityScript:IsLocal()
	return self.entity:IsLocal()
end

function EntityScript:IsMinimal()
	return self.entity:IsMinimal()
end

function EntityScript:IsLocalOrMinimal()
	return self.entity:IsLocalOrMinimal()
end

function EntityScript:IsAncestorsLocal()
	local parent = self.entity:GetParent()
	while parent do
		if parent:IsNetworked() and not parent:IsLocal() then
			if SuperVerboseNetworkLogging then
				TheLog.ch.Networking:printf("Entity %s has a non-local network ancestor %s", self, parent)
			end
			return false
		end
		parent = parent.entity:GetParent()
	end
	return true
end

function EntityScript:ShouldSendNetEvents()
	return self:IsLocal() and self:IsNetworked() and not self:IsMinimal()
end

function EntityScript:IsTransferable()
	return self.entity:IsTransferable()
end

function EntityScript:CanTakeControl(attack)
	if self:IsLocal() then
		return true
	end
	if not self:IsTransferable() then
		return false
	end

	local health_cmp = self.components.health
	if health_cmp and health_cmp:GetCurrent() <= 1 then
		return false
	end

	if self.HitBox and self.HitBox:IsInvincible() then
		return false
	end

	-- The order here might be important. Checking this after IsInvicible check because if it's invincible, the attack won't connect.
	if attack and health_cmp and self.components.combat then
		-- If we predict that this attack will kill the entity, take control so the kill feels local.
		local damage = self.components.combat:CalculateProcessedDamage(attack)
		if self:IsAlive() and damage >= health_cmp:GetCurrent() and not self:HasTag("no_state_transition") then
			return true
		end
	end

	if self.sg then
		return self.sg:CanTakeControl() or self.sg:CanTakeControlByKnockingAttack(attack)
	end
	return true
end

function EntityScript:TryToTakeControl()
	if self:CanTakeControl() then
		self:TakeControl()
		return true
	end
	return false
end

function EntityScript:TakeControl()
	self.entity:TakeControl()
end

function EntityScript:PushControlToHost()
	self.entity:PushControlToHost()
end

function EntityScript:IsInLimbo()
	return self.entity:HasTag("INLIMBO")
end

-------------------------------------------------
-- Life Status Functions
function EntityScript:IsAlive()
	return self.components.health ~= nil and self.components.health:IsAlive()
end

function EntityScript:IsDying()
	return self.components.health ~= nil and self.components.health:IsDying()
end

function EntityScript:IsDead()
	return self.components.health ~= nil and self.components.health:IsDead()
end

function EntityScript:IsRevivable()
	return self.components.health ~= nil and self.components.health:IsRevivable()
end
-------------------------------------------------

function EntityScript:RemoveFromScene()
	self.entity:AddTag("INLIMBO")
	self:ResolveInLimboTag()
end

function EntityScript:ReturnToScene()
	self.entity:RemoveTag("INLIMBO")
	self:ResolveInLimboTag()
end

function EntityScript:Pause()
	self.paused = true
	if self.Physics ~= nil then
		self.Physics:Pause()
	end
	if not self:IsInLimbo() and self.AnimState ~= nil then
		self.AnimState:Pause()
	end
	if self.sg ~= nil then
		self.sg:Pause("entitypause")
	end
	self:PushEvent("paused")
end

function EntityScript:Resume()
	self.paused = false
	if self.Physics ~= nil then
		self.Physics:Resume()
	end
	if not self:IsInLimbo() and self.AnimState ~= nil then
		self.AnimState:Resume()
	end
	if self.sg ~= nil then
		self.sg:Resume("entitypause")
	end
	self:PushEvent("resumed")
end

-- Turn off decision making, but not stategraph or animations.
function EntityScript:Stupify(reason)
	assert(reason, "Give a reason why you stupify")
	if self.brain then
		self.brain:Pause(reason)
	end
	if self.boss_coro then
		self.boss_coro:Stop(reason)
	end
end

function EntityScript:Unstupify(reason)
	assert(reason, "Use the same reason passed to Stupify().")
	if self.brain then
		self.brain:Resume(reason)
	end
	if self.boss_coro then
		self.boss_coro:Resume(reason)
	end
end

function EntityScript:__tostring()
	local i = self.GetHunterId and self:GetHunterId()
	local label = i and " P".. i or ""

	-- self.name is the string passed into CreateEntity()
	return string.format("%d - %s%s%s", self.GUID, self.name or self.prefab or "", label, self._netisinlimbo and "(LIMBO)" or "")
end

function EntityScript:GetTimeAlive()
	return GetTime() - self.spawntime
end

function EntityScript:StartUpdatingComponent(cmp)
	if StopUpdatingComponents[cmp] ~= nil then
		StopUpdatingComponents[cmp] = nil
	elseif self.updatecomponents == nil or not self.updatecomponents[cmp] then
		if self.newupdatecomponents == nil then
			self.newupdatecomponents = { [cmp] = true }
			NewUpdatingEnts[self] = true
		else
			self.newupdatecomponents[cmp] = true
		end
	end
end

function EntityScript:StartUpdatingComponents_Deferred()
	if self.updatecomponents == nil then
		self.updatecomponents = self.newupdatecomponents
		UpdatingEnts[self] = true
	else
		for cmp in pairs(self.newupdatecomponents) do
			self.updatecomponents[cmp] = true
		end
	end
	self.newupdatecomponents = nil
	NewUpdatingEnts[self] = nil
end

function EntityScript:StopUpdatingComponent(cmp)
	if self.updatecomponents ~= nil and self.updatecomponents[cmp] then
		StopUpdatingComponents[cmp] = self
	elseif self.newupdatecomponents ~= nil and self.newupdatecomponents[cmp] then
		self.newupdatecomponents[cmp] = nil
		if next(self.newupdatecomponents) == nil then
			self.newupdatecomponents = nil
			NewUpdatingEnts[self] = nil
		end
	end
end

function EntityScript:StopAllUpdatingComponents()
	if self.updatecomponents ~= nil then
		for cmp in pairs(self.updatecomponents) do
			StopUpdatingComponents[cmp] = self
		end
	end
	if self.newupdatecomponents ~= nil then
		self.newupdatecomponents = nil
		NewUpdatingEnts[self] = nil
	end
end

function EntityScript:StopUpdatingComponent_Deferred(cmp)
	StopUpdatingComponents[cmp] = nil
	self.updatecomponents[cmp] = nil
	if next(self.updatecomponents) == nil then
		self.updatecomponents = nil
		UpdatingEnts[self] = nil
	end
end

function EntityScript:StartWallUpdatingComponent(cmp)
	if StopWallUpdatingComponents[cmp] ~= nil then
		StopWallUpdatingComponents[cmp] = nil
	elseif self.wallupdatecomponents == nil or not self.wallupdatecomponents[cmp] then
		if self.newwallupdatecomponents == nil then
			self.newwallupdatecomponents = { [cmp] = true }
			NewWallUpdatingEnts[self] = true
		else
			self.newwallupdatecomponents[cmp] = true
		end
	end
end

function EntityScript:StartWallUpdatingComponents_Deferred()
	if self.wallupdatecomponents == nil then
		self.wallupdatecomponents = self.newwallupdatecomponents
		WallUpdatingEnts[self] = true
	else
		for cmp in pairs(self.newwallupdatecomponents) do
			self.wallupdatecomponents[cmp] = true
		end
	end
	self.newwallupdatecomponents = nil
	NewWallUpdatingEnts[self] = nil
end

function EntityScript:StopWallUpdatingComponent(cmp)
	if self.wallupdatecomponents ~= nil and self.wallupdatecomponents[cmp] then
		StopWallUpdatingComponents[cmp] = self
	elseif self.newwallupdatecomponents ~= nil and self.newwallupdatecomponents[cmp] then
		self.newwallupdatecomponents[cmp] = nil
		if next(self.newwallupdatecomponents) == nil then
			self.newwallupdatecomponents = nil
			NewWallUpdatingEnts[self] = nil
		end
	end
end

function EntityScript:StopAllWallUpdatingComponents()
	if self.wallupdatecomponents ~= nil then
		for cmp in pairs(self.wallupdatecomponents) do
			StopWallUpdatingComponents[cmp] = self
		end
	end
	if self.newwallupdatecomponents ~= nil then
		self.newwallupdatecomponents = nil
		NewWallUpdatingEnts[self] = nil
	end
end

function EntityScript:StopWallUpdatingComponent_Deferred(cmp)
	StopWallUpdatingComponents[cmp] = nil
	self.wallupdatecomponents[cmp] = nil
	if next(self.wallupdatecomponents) == nil then
		self.wallupdatecomponents = nil
		WallUpdatingEnts[self] = nil
	end
end


function EntityScript:GetComponentName(cmp)
	for k, v in pairs(self.components) do
		if v == cmp then
			return k
		end
	end
	return "component"
end

-- == A Partial List of Tags ==
-- FX: Animated visual effects. Makes raycasts return parent entity and is ignored by GetEntitiesOnTileAtPoint.
-- INLIMBO: Entity exists, but isn't in the current scene (mostly inactive).
-- NOCLICK: Do not allow the mouse to click.
-- YESCLICK: Override optimizations that would exclude this object from being clicked.
-- interactable: Entities the player can interact with.
-- player: This is a player.
-- prop: Items placeable in the level by an artist.
-- survives_room_travel: Not destroyed when moving between rooms in a dungeon.
function EntityScript:AddTag(tag)
	self.entity:AddTag(tag)
end

function EntityScript:RemoveTag(tag)
	self.entity:RemoveTag(tag)
end

function EntityScript:HasTag(tag)
	return self.entity:HasTag(tag)
end

-- Call to prevent an entity from being destroyed when we move between rooms in
-- a dungeon.
--
-- Don't call this unless you're 100% sure we need to keep this entity around.
-- Don't expect that this state can be removed!
function EntityScript:MakeSurviveRoomTravel()
	self:AddTag("survives_room_travel")
	-- OnMakeSurviveRoomTravel may be called *twice*: here and in PostSpawn.
	-- Most components are added after MakeSurviveRoomTravel, but if tagged
	-- after spawn we apply here to ensure all components receives the message.
	-- Components added after spawn are not guaranteed to receive it.
	for cmp_name, cmp in pairs(self.components) do
		if cmp.OnMakeSurviveRoomTravel then
			-- Implementations should only call MakeSurviveRoomTravel on their
			-- child entities.
			cmp:OnMakeSurviveRoomTravel()
		end
	end
	return self
end

function EntityScript:OnRoomTravel()
	for _cmp_name, cmp in pairs(self.components) do
		if cmp.OnRoomTravel then
			cmp:OnRoomTravel()
		end
	end
end

-- For entities created by debug tools. Add all the tags to make everything
-- ignore them.
function EntityScript:TagAsDebugTool()
	self:AddTag("dbg_tool")
	self:AddTag("dbg_nohistory")
	return self
end

function EntityScript:AddComponent(name, ...)
	local cmp = self.components[name]
	if cmp ~= nil then
		print("component "..name.." already exists on entity "..tostring(self).."!"..debugstack_oneline(3))
	else
		cmp = LoadComponent(name)
		assert(cmp, "component "..name.." does not exist!")
		cmp = cmp(self, ...)
		self.components[name] = cmp
	end

	if cmp.OnNetSerialize ~= nil and cmp.OnNetDeserialize ~= nil
		and (not self.prefab or Prefabs[self.prefab].network_type ~= NetworkType_None)
	then
		self.netcomponents[smallhash(name)] = cmp
		self.nrnetcomponents = self.nrnetcomponents + 1

		NetworkUpdatingEnts[self] = true
	end

	--~ if DEV_MODE then
	--~ 	cmp.__zone_update_component = string.format("%s:OnUpdate", name)
	--~ 	cmp.__zone_serialize_netcomponent = string.format("%s:NetSerialize", name)
	--~ end
	return cmp
end

function EntityScript:RemoveComponent(name)
	local cmp = self.components[name]
	if cmp ~= nil then

		if cmp.OnNetSerialize ~= nil and cmp.OnNetDeserialize ~= nil then
			self.netcomponents[smallhash(name)] = nil
			self.nrnetcomponents = self.nrnetcomponents - 1

			if self.nrnetcomponents == 0 then
				NetworkUpdatingEnts[self] = nil
			end
		end

		self:StopUpdatingComponent(cmp)
		self:StopWallUpdatingComponent(cmp)
		self.components[name] = nil
		if cmp.OnRemoveFromEntity ~= nil then
			cmp:OnRemoveFromEntity()
		end
	end
end

function EntityScript:GetDisplayName()
	return STRINGS.NAMES[self.prefab]
end

function EntityScript:SetPrefabName(name)
	dbassert(Prefabs[name], "Setting an unknown prefab name. Did you mean inst.entity:SetName()?")
	self.prefab = name
	self.entity:SetPrefabName(name)
end

function EntityScript:GetBrainString()
	if self.brain == nil then
		return ""
	end
	return "Brain:\n"
		..self.brain:GetDebugString()
		.."--------\n"
end

function EntityScript:GetDebugString()
	if not self:IsValid() then
		return tostring(self).." <INVALID>"
	end

	local str = tostring(self)
	if self:IsAsleep() then
		str = str.." <ASLEEP>"
	end
	if self:IsInLimbo() then
		str = str.." <Limbo>"
	end
	if not self:IsVisible() then
		str = str.." <Not Visible>"
	end
	str = str..string.format(" age %2.2f\n", self:GetTimeAlive())
	str = str..self.entity:GetDebugString()
	if not self.Network then
		-- Component prints iff it exists, print absence to be clear.
		str = str .. "[no Network]"
	end
	if self.sg ~= nil then
		str = str.."SG: ".. self.sg:GetDebugString() .."\n"
	end
	if self.debugstringfn ~= nil then
		str = str.."DebugString: "..self:debugstringfn().."\n"
	end
	str = str.."-----------\n"

	local cmpnames = {}
	for k in pairs(self.components) do
		cmpnames[#cmpnames + 1] = k
	end
	table.sort(cmpnames)
	for i = 1, #cmpnames do
		local cmpname = cmpnames[i]
		local cmp = self.components[cmpname]
		if cmp.GetDebugString ~= nil then
			local cmpstr = cmp:GetDebugString()
			if cmpstr ~= nil then
				str = str..cmpname..": "..cmpstr.."\n"
			end
		end
	end
	return str
end

-- Wrap native components (userdata) with a table so you can capture certain
-- calls. I expect this to be used as a debug feature and not a normal gameplay
-- feature, so I'm not sure it's 100% reliable.
--
-- After wrapping, you can define lua functions to wrap the native ones:
--   inst:Debug_WrapNativeComponent("Physics")
--   inst.Physics.SetVel = function(self, x, y, z)
--     if x == 0 then
--       print(self.inst, debug.traceback())
--     end
--     DebugDraw.GroundRay_Vec(inst:GetPosition(), Vector3(x,y,z), WEBCOLORS.YELLOW)
--     self._original:SetVel(x, y, z)
--   end
-- Call inst.Physics:UnwrapNativeComponent() to undo.
--
-- See also instrument_userdata().
function EntityScript:Debug_WrapNativeComponent(comp_name)
	local comp = self[comp_name]
	assert(comp, "Wrapping nonexistent component.")
	assert(not comp.UnwrapNativeComponent, "Component is already wrapped.")
	local Proxy = {
		inst = self,
		UnwrapNativeComponent = function()
			self[comp_name] = comp
		end,
		_original = comp,
	}
	function Proxy.__index(t, k)
		local fn = function(_self, ...)
			return comp[k](comp, ...)
		end
		Proxy[k] = fn
		return fn
	end
	setmetatable(Proxy, Proxy)
	self[comp_name] = Proxy
end


function EntityScript:SetBossCoro(name)
	if self.boss_coro then
		return
	end
	local coro = LoadBossCoro(name)
	self.boss_coro = coro(self)
end

function EntityScript:SetBrain(name)
	if self.brain ~= nil then
		self.brain:OnRemoveFromEntity()
		self.brain = nil
	end
	if name ~= nil then
		self.brain = BrainInstance(self, LoadBrain(name))
		if self:IsInLimbo() then
			self.brain:Pause("limbo")
		end
		if self:IsAsleep() then
			self.brain:Pause("entitysleep")
		end

		if not self:IsLocalOrMinimal() then
			self.brain:Pause("remote")
		end

		self.brain:Start()
	end
end

function EntityScript:SetEmbellisherPrefab(name)
	self.embellisher_prefab_override = name
end

function EntityScript:SetStateGraph(name, opt_table)
	if self.sg ~= nil then
		self.sg:OnRemoveFromEntity()
		self.sg = nil
	end
	if name then
		if (opt_table) then
			self.sg = StateGraphInstance(self, opt_table)
		else
			local prefabname = self.embellisher_prefab_override or self.prefab
			local entityName = self:HasTag("widget") and self.name or prefabname
			assert(entityName, [[First call SetPrefabName on your prefab:
				local function fn(prefabname)
					local inst = CreateEntity()
					inst:SetPrefabName(prefabname)
			]])
			self.sg = StateGraphInstance(self, LoadStateGraph(name, entityName))
		end

		local def = embellishutil.GetEmbellishmentForPrefab(self)
		if def and def.embellishments then
			if def.needSoundEmitter and not self.SoundEmitter then
				self.entity:AddSoundEmitter()
			end
			self.sg:Embellish(def.embellishments)
		end

		if self:IsInLimbo() then
			self.sg:Pause("limbo")
		end
		if self:IsAsleep() then
			self.sg:Pause("entitysleep")
		end
		if not self:IsLocalOrMinimal() then
			self.sg:Pause("remote")
		end

		self.sg:Start()
	end
end

local function AddEventListenerFn(t, event, inst, fn)
	local listeners = t[event]
	if listeners == nil then
		listeners = {}
		t[event] = listeners
	end

	local fns = listeners[inst]
	if fns == nil then
		fns = {}
		listeners[inst] = fns
	end

	fns[#fns + 1] = fn
end

function EntityScript:ListenForEvent(event, fn, source)
	source = source or self
	fn = { fn = fn } --container so we can invalidate it even when cached during PushEvent

	if source.eventlisteners == nil then
		source.eventlisteners = {}
	end
	AddEventListenerFn(source.eventlisteners, event, self, fn)

	if self.eventlistening == nil then
		self.eventlistening = {}
	end
	AddEventListenerFn(self.eventlistening, event, source, fn)
end

-- Returns the removed listener container and whether t is completely empty.
local function RemoveEventListenerFn(t, event, inst, fn)
	local listeners = t and t[event]
	if not listeners then
		return
	end
	local fns = listeners[inst]
	if not fns then
		return
	end
	for i = 1, #fns do
		local c = fns[i]
		if fn == c.fn then
			-- Removes empty tables to reduce memory high watermark?
			if #fns == 1 then
				listeners[inst] = nil
				local was_last_listener_for_event = next(listeners) == nil
				if was_last_listener_for_event then
					t[event] = nil
					return c, next(t) == nil
				end
			else
				table.remove(fns, i)
			end
			return c, false
		end
	end
end

function EntityScript:RemoveEventCallback(event, fn, source)
	source = source or self

	local c, empty = RemoveEventListenerFn(self.eventlistening, event, source, fn)
	-- c is the container for fn, if found
	if c ~= nil then
		if empty then
			self.eventlistening = nil
		end

		c, empty = RemoveEventListenerFn(source.eventlisteners, event, self, c.fn)
		if empty then
			source.eventlisteners = nil
		end

		c.fn = nil --invalidate in case we're called from PushEvent
	end
end

local function RemoveEventListener(t, event, inst)
	local listeners = t[event]
	listeners[inst] = nil
	if next(listeners) == nil then
		t[event] = nil
		return next(t) == nil
	end
	return false
end

function EntityScript:RemoveAllEventCallbacks()
	if self.eventlistening ~= nil then
		for event, sources in pairs(self.eventlistening) do
			for source, fns in pairs(sources) do
				for i = 1, #fns do
					fns[i].fn = nil --invalidate in case we're called from PushEvent
				end
				if RemoveEventListener(source.eventlisteners, event, self) then
					source.eventlisteners = nil
				end
			end
		end
		self.eventlistening = nil
	end

	if self.eventlisteners ~= nil then
		for event, listeners in pairs(self.eventlisteners) do
			for listener, fns in pairs(listeners) do
				for i = 1, #fns do
					fns[i].fn = nil --invalidate in case we're called from PushEvent
				end
				if RemoveEventListener(listener.eventlistening, event, self) then
					listener.eventlistening = nil
				end
			end
		end
		self.eventlisteners = nil
	end
end

function EntityScript:PushEvent(event, data)
	if self.eventlisteners ~= nil then
		local listeners = self.eventlisteners[event]
		if listeners ~= nil then
			-- Build local list since eventlisteners might change during
			-- iteration.
			local tocall = {}
			local n = 0
			for _, fns in pairs(listeners) do
				for i = 1, #fns do
					n = n + 1
					tocall[n] = fns[i]
				end
			end
			for i = 1, n do
				local c = tocall[i]
				if c.fn ~= nil then --make sure the callback wasn't cleared (RemoveEventCallback)
					c.fn(self, data)
				end
			end
		end
	end

	if self.sg ~= nil then
		self.sg:PushEvent(event, data)
	end

	if self.brain ~= nil then
		self.brain:PushEvent(event, data)
	end
end

function EntityScript:GetPosition()
	return Point(self.Transform:GetWorldPosition())
end

function EntityScript:GetAngleTo(other)
	return self:GetAngleToXZ(other.Transform:GetWorldXZ())
end

function EntityScript:GetAngleToPoint(pt)
	return self:GetAngleToXZ(pt:GetXZ())
end

function EntityScript:GetAngleToXZ(x, z)
	local x1, z1 = self.Transform:GetWorldXZ()
	if x == x1 and z == z1 then
		return self.Transform:GetRotation()
	end
	return math.deg(math.atan(z1 - z, x - x1))
end

function EntityScript:IsWithinAngleTo(other, angle, angle_max)
	if not other then
		return false
	end

	local x, z = other.Transform:GetWorldXZ()
	return self:IsWithinAngleToXZ(x, z, angle, angle_max)
end

function EntityScript:IsWithinAngleToPoint(pt, angle, angle_max)
	local x, z = pt:GetXZ()
	return self:IsWithinAngleToXZ(x, z, angle, angle_max)
end

-- Checks if an entity's angle towards a point is within the specifed angles in world space.
function EntityScript:IsWithinAngleToXZ(x, z, angle_a, angle_b)
	local angle_to_pt = self:GetAngleToXZ(x, z)
	local angle = SimplifyAngle(angle_to_pt)
	angle_b = angle_b or -angle_a
	local min = SimplifyAngle(angle_a)
	local max = SimplifyAngle(angle_b)

	-- TODO: make a more elegant solution...
	if min > max then
		return angle >= min or angle <= max
	else
		return angle > min and angle < max
	end
end

function EntityScript:GetPositionAdjacentTo(target, distance)
	if target == nil then
		return nil
	end
	local p1 = Vector3(self.Transform:GetWorldPosition())
	local p2 = Vector3(target.Transform:GetWorldPosition())
	local offset = p1-p2
	offset:Normalize()
	offset = offset * distance
	return (p2 + offset)
end

function EntityScript:Face(other)
	self:FaceXZ(other.Transform:GetWorldXZ())
end

function EntityScript:FacePoint(pt)
	self:FaceXZ(pt:GetXZ())
end

function EntityScript:FaceXZ(x, z)
	local x1, z1 = self.Transform:GetWorldXZ()
	if x == x1 and z == z1 then
		return
	end
	self.Transform:SetRotation(math.deg(math.atan(z1 - z, x - x1)))
end

function EntityScript:FaceAwayFrom(other)
	self:FaceAwayFromXZ(other.Transform:GetWorldXZ())
end

function EntityScript:FaceAwayFromPoint(pt)
	self:FaceAwayFromXZ(pt:GetXZ())
end

function EntityScript:FaceAwayFromXZ(x, z)
	local x1, z1 = self.Transform:GetWorldXZ()
	if x == x1 and z == z1 then
		return
	end
	self.Transform:SetRotation(math.deg(math.atan(z - z1, x1 - x)))
end

function EntityScript:FlipFacingAndRotation()
	--Make sure facing flips even when rotated directly up or down
	self.Transform:FlipFacingAndRotation()
end

function EntityScript:SnapToFacingRotation()
	self.Transform:SetRotation(self.Transform:GetFacingRotation())
end

function EntityScript:GetDistanceSqTo(other)
	local x1, z1 = self.Transform:GetWorldXZ()
	local x2, z2 = other.Transform:GetWorldXZ()
	return DistSq2D(x1, z1, x2, z2)
end

function EntityScript:GetDistanceSqToPoint(pt)
	return self:GetDistanceSqToXZ(pt:GetXZ())
end

function EntityScript:GetDistanceSqToXZ(x, z)
	local x1, z1 = self.Transform:GetWorldXZ()
	return DistSq2D(x, z, x1, z1)
end

function EntityScript:IsNear(other, dist)
	return self:GetDistanceSqTo(other) < dist * dist
end

function EntityScript:IsNearPoint(pt, dist)
	return self:GetDistanceSqToXZ(pt:GetXZ()) < dist * dist
end

function EntityScript:IsNearXZ(x, z, dist)
	return self:GetDistanceSqToXZ(x, z) < dist * dist
end

function EntityScript:IsNearPlayer(range, isalive)
	local x, z = self.Transform:GetWorldXZ()
	return playerutil.IsAnyPlayerInRange(x, z, range, isalive)
end

function EntityScript:GetClosestPlayer(isalive)
	local x, z = self.Transform:GetWorldXZ()
	--returns player, distsq
	return playerutil.FindClosestPlayer(x, z, isalive)
end

function EntityScript:GetClosestPlayerInRange(range, isalive)
	local x, z = self.Transform:GetWorldXZ()
	--returns player, distsq
	return playerutil.FindClosestPlayerInRange(x, z, range, isalive)
end

function EntityScript:GetRandomEntityByTagInRange(range, tags, isalive, checkinvisible)
	local x, z = self.Transform:GetWorldXZ()
	--returns player, distsq
	local ents = TheSim:FindEntitiesXZ(x, z, range, nil, {"INLIMBO"}, tags)
	local valid_ents = {}

	for i, ent in ipairs(ents) do
		if (isalive == nil or isalive == ent:IsAlive()) and (ent:IsVisible() or checkinvisible) and ent ~= self then
			table.insert(valid_ents, ent)
		end
	end

	if #valid_ents > 0 then
		return valid_ents[math.random(#valid_ents)]
	end
end

function EntityScript:GetClosestEntityByTagInRange(range, tags, isalive, iscombattarget, checkinvisible)
	local x, z = self.Transform:GetWorldXZ()
	--returns player, distsq
	local ents = TheSim:FindEntitiesXZ(x, z, range, nil, {"INLIMBO"}, tags)
	local closest
	local closest_sqdist
	for i, ent in ipairs(ents) do
		if (isalive == nil or isalive == ent:IsAlive()) and (ent:IsVisible() or checkinvisible) and ent ~= self then
			local valid_target = true
			if iscombattarget then
				local target_sg = ent.sg
				valid_target = not target_sg or (target_sg and not target_sg:HasStateTag("notarget"))
			end

			local distsq = ent:GetDistanceSqToXZ(x, z)
			if valid_target and (closest_sqdist == nil or distsq < closest_sqdist) then
				closest_sqdist = distsq
				closest = ent
			end
		end
	end

	return closest, closest ~= nil and closest_sqdist or nil
end


function EntityScript:CancelAllTasks()
	if self.tasks ~= nil then
		for k in pairs(self.tasks) do
			k:Cancel()
		end
	end
end

function EntityScript:OnCancelTask(task)
	self.tasks[task] = nil
end

function EntityScript:AddTaskInternal(task)
	task.inst = self --This is so the task can trigger OnCancelTask
	if self.tasks == nil then
		self.tasks = { [task] = true }
	else
		self.tasks[task] = true
	end
	return task
end

function EntityScript:DoPeriodicTask(period, fn, initialdelay, ...)
	return self:AddTaskInternal(Scheduler:ExecutePeriodic(period, fn, nil, initialdelay, ...))
end

function EntityScript:DoPeriodicTicksTask(period, fn, initialdelay, ...)
	return self:AddTaskInternal(Scheduler:ExecutePeriodicTicks(period, fn, nil, initialdelay, ...))
end

-- Do a task that updates every tick with a progress value.
--
-- First update is *in the current tick* and last one is `ticks` later so
-- DoDurationTaskForTicks(10, fn) will see fn called 11 times. The first with
-- progress=0 and last with progress=1.
function EntityScript:DoDurationTaskForTicks(ticks, fn, initialdelay, ...)
	initialdelay = initialdelay or 0
	local ticks_elapsed = -1
	local updatefn = function(inst_)
		ticks_elapsed = ticks_elapsed + 1
		local progress = ticks_elapsed / ticks
		fn(self, progress)
	end
	-- Run immediately to ensure initial state is correct. Scheduler doesn't
	-- support running on the current frame.
	updatefn(self)
	return self:AddTaskInternal(Scheduler:ExecutePeriodicTicks(1, updatefn, ticks, initialdelay, ...))
end

-- Do a task that updates every tick with a progress value, but with timing
-- based on animframes. Usually, we want timing applied per tick for higher
-- fidelity results, but use DoPeriodicTicksTask(ANIM_FRAMES, fn) if you
-- really want only one update per anim frame instead.
--
-- The first call will happen *in the current tick* with progress=0 and last
-- `duration` frames later with progress=1.
function EntityScript:DoDurationTaskForAnimFrames(duration, fn, initialdelay, ...)
	local ticks = duration * ANIM_FRAMES
	initialdelay = (initialdelay or 0) * ANIM_FRAMES
	return self:DoDurationTaskForTicks(ticks, fn, initialdelay, ...)
end

function EntityScript:DoTaskInTime(time, fn, ...)
	return self:AddTaskInternal(Scheduler:ExecuteInTime(time, fn, ...))
end

function EntityScript:DoTaskInTicks(ticks, fn, ...)
	return self:AddTaskInternal(Scheduler:ExecuteInTicks(ticks, fn, ...))
end

-- Use a timescale that matches standard anim playback speed (instead of update
-- ticks). Does *not* scale with anim slomo.
function EntityScript:DoTaskInAnimFrames(animframes, fn, ...)
	return self:AddTaskInternal(Scheduler:ExecuteInTicks(animframes * ANIM_FRAMES, fn, ...))
end

function EntityScript:StartThread(fn, ...)
	return self:AddTaskInternal(Scheduler:StartThread(fn, ...))
end

-- Use case: Allow the network to account for latency between "death" events and actual removal
function EntityScript:DelayedRemove()
	if not self.delayedremovetask then
		self:RemoveFromScene()
		self.delayedremovetask = self:DoTaskInTime(1, function() self:Remove() end)
		if self.Network and SuperVerboseNetworkLogging then
			TheLog.ch.Networking:printf("Entity GUID %d EntityID %d Delayed Remove", self.GUID, self.Network:GetEntityID())
		end
	end
end

function EntityScript:IsInDelayedRemove()
	return self.delayedremovetask ~= nil
end

function EntityScript:CancelDelayedRemove()
	if self.delayedremovetask then
		self.delayedremovetask:Cancel()
		self.delayedremovetask = nil
		if self.Network and SuperVerboseNetworkLogging then
			TheLog.ch.Networking:printf("Entity GUID %d EntityID %d Delayed Remove Cancelled.", self.GUID, self.Network:GetEntityID())
		end
		self:ReturnToScene()
	end
end

function EntityScript:Remove(forceremove)
	-- used to allow native calls to forcibly remove entities and preserve dev/editor behaviour
	forceremove = forceremove or IsLocalGame

	if not forceremove and self.Network then
		if not self:IsLocal() then
			if SuperVerboseNetworkLogging then
				TheLog.ch.Networking:printf("Disallowing removal of remote Entity GUID %d EntityID %d.", self.GUID, self.Network:GetEntityID())
			end
			if self.delayedremovetask then
				self.delayedremovetask:Cancel()
				self.delayedremovetask = nil
			end
			return
		end
		-- Default to using a delayed remove for network entities
		-- Disallow immediate removals if a delayed remove is not done
		if not self.delayedremovetask then
			self:DelayedRemove()
			return
		elseif not self.delayedremovetask:IsDone() then
			if SuperVerboseNetworkLogging then
				TheLog.ch.Networking:printf("Disallowing immediate removal of Entity GUID %d EntityID %d with Delayed Remove.", self.GUID, self.Network:GetEntityID())
			end
			return
		end
		self.delayedremovetask = nil
	else
		if self.delayedremovetask then
			if SuperVerboseNetworkLogging and not self.delayedremovetask:IsDone() then
				TheLog.ch.Networking:printf("Immediately removing Entity GUID %d with a delayed remove.", self.GUID)
			end
			self.delayedremovetask:Cancel()
			self.delayedremovetask = nil
		end
	end

	if GetDebugEntity() == self then
		SetDebugEntity(nil)
	end

	self:PushEvent("onremove")

	for k, v in pairs(self.components) do
		if v.OnRemoveEntity ~= nil then
			v:OnRemoveEntity()
		end
	end

	if self.OnRemoveEntity ~= nil then
		self:OnRemoveEntity()
	end

	self:SetBrain(nil)
	self:SetStateGraph(nil)
	self:RemoveAllEventCallbacks()
	self:CancelAllTasks()
	self:StopAllUpdatingComponents()
	self:StopAllWallUpdatingComponents()

	if self.Network and SuperVerboseNetworkLogging then
		TheLog.ch.Networking:printf("Entity GUID %d EntityID %d Removed.", self.GUID, self.Network:GetEntityID())
	end

	Ents[self.GUID] = nil
	PhysicsCollisionCallbacks[self.GUID] = nil

	NetworkUpdatingEnts[self] = nil

	self.persists = false
	self.entity:Retire()
end

function EntityScript:GetPersistData()
	local data

	for k, v in pairs(self.components) do
		if v.OnSave ~= nil then
			local t = v:OnSave()
			--Don't save if t is nil, false, or an empty table
			if t and (type(t) ~= "table" or next(t) ~= nil) then
				if data == nil then
					data = { [k] = t }
				else
					data[k] = t
				end
			end
		end
	end

	if self.OnSave ~= nil then
		if data == nil then
			data = {}
		end
		self:OnSave(data)
		if next(data) == nil then
			data = nil
		end
	end

	return data
end

-- Called at the end of SpawnPrefab, prefab is fully initalized at this point and all other components/ tags etc. are present.
function EntityScript:PostSpawn()
	for cmp_name, cmp in pairs(self.components) do
		if cmp and cmp.OnPostSpawn then
			cmp:OnPostSpawn()
		end
	end

	if self.OnPostSpawn then
		self:OnPostSpawn()
	end

	if self:HasTag("survives_room_travel") then
		-- Repeat this callback in post spawn so components added after initial
		-- tag get notified.
		self:MakeSurviveRoomTravel()
	end
end

-- After the world completes loading or an entity is spawned after the world is
-- already loaded, fire this callback to allow additional construction logic.
-- We always call OnPostLoadWorld so both new and saved objects can do post
-- spawn fixup.
function EntityScript:PostLoadWorld(data)
	for cmp_name, cmp in pairs(self.components) do
		if cmp and cmp.OnPostLoadWorld then
			local savedata = data and data[cmp_name] or nil -- nil for new spawns
			cmp:OnPostLoadWorld(savedata)
		end
	end

	if self.OnPostLoadWorld then
		self:OnPostLoadWorld(data)
	end
end

function EntityScript:SetPersistData(data)
	--print("SetPersistData called on entity " .. self.entity:GetPrefabName())

	if self.OnPreLoad ~= nil then
		self:OnPreLoad(data)
	end

	if data ~= nil then
		for k, v in pairs(data) do
			local cmp = self.components[k]
			if cmp ~= nil and cmp.OnLoad ~= nil then
				cmp:OnLoad(v)
			end
		end
	end
	if self.OnLoad ~= nil then
		self:OnLoad(data)
	end
end

function EntityScript:Embellish()
	local def = embellishutil.GetEmbellishmentForPrefab(self)
	if def and def.embellishments then
		if def.needSoundEmitter and not self.SoundEmitter then
			self.entity:AddSoundEmitter()
		end
		if not self.sg then
			self:SetStateGraph(self.prefab, GenerateStateGraph(self.sgname_override or self.prefab))
			self.sg:Embellish(def.embellishments)
		end
	end
end

function EntityScript:UncacheStateGraph()
	if self.sg and self.sg.sg then
		UncacheStateGraph(self.sg.sg.name, self.prefab)
	end
end

--#V2C: deprecate this?
function EntityScript:LongUpdate(dt)
	if self.OnLongUpdate ~= nil then
		self:OnLongUpdate(dt)
	end

	for k, v in pairs(self.components) do
		if v.LongUpdate ~= nil then
			v:LongUpdate(dt)
		end
	end
end

-- utility predicate for table.sort
function EntityScript.OrderByXZDistanceFromOrigin(a,b)
	local ax, az = a.Transform:GetWorldXZ()
	local adistsq = ax * ax + az * az
	local bx, bz = b.Transform:GetWorldXZ()
	local bdistsq = bx * bx + bz * bz
	return adistsq < bdistsq
end

-- network permanent flag API

function EntityScript:SetPermanentFlags(flags)
	self.entity:SetPermanentFlags(flags)
end

function EntityScript:GetPermanentFlags()
	return self.entity:GetPermanentFlags()
end

function EntityScript:SetIgnorePermanentFlagChanges(should_ignore, flags)
	flags = flags or PFLAG_ALL
	self.ignore_pflags = self.ignore_pflags or 0
	if should_ignore then
		self.ignore_pflags = self.ignore_pflags | flags
	else
		self.ignore_pflags = self.ignore_pflags & ~flags
	end
end

function EntityScript:GetIgnorePermanentFlagChanges()
	return self.ignore_pflags or 0
end

local MaxNrComponentsBits <const> = 8


function EntityScript:NetSerialize()
	-- Save the components that need saving:
	if self.entity:NeedsSerializing() then
	--	print("serializing " .. self.entity:GetPrefabName())

		self.entity:StartSerialization()

		if self.owner and self.owner:IsValid() and self.owner:IsNetworked() then
			self.entity:SerializeBoolean(true)
			self.entity:SerializeEntityID(self.owner.Network:GetEntityID());
		else
			self.entity:SerializeBoolean(false)
		end

		-- Serialize the stategraph if it exists:
		if self.sg then
			--~ TheSim:ProfilerPush("networkserialize.sg")
			self.entity:PushSerializationMarker("sg");
			self.sg:OnNetSerialize()
			self.entity:PopSerializationMarker()
			--~ TheSim:ProfilerPop()
		end

		if self.boss_coro and self.boss_coro.OnNetSerialize then
			--~ TheSim:ProfilerPush("networkserialize.boss_coro")
			self.entity:PushSerializationMarker("boss_coro");
			self.boss_coro:OnNetSerialize()
			self.entity:PopSerializationMarker()
			--~ TheSim:ProfilerPop()
		end

		self.entity:SerializeUInt(self.nrnetcomponents, MaxNrComponentsBits)

		for k, v in pairs(self.netcomponents) do
			--~ TheSim:ProfilerPush(v.__zone_serialize_netcomponent or "Component:NetSerialize")

			-- save the hash of the name of the netcomponent:
			self.entity:AlignToByte();

			local compname = GetComponentNameForHash(k)
			if compname then
				self.entity:PushSerializationMarker( compname );
			end

			self.entity:SerializeUInt(k, 32);	-- 32 bit hash as uint32

			v:OnNetSerialize()

			if compname then
				self.entity:PopSerializationMarker();
			end

			--~ TheSim:ProfilerPop()
		end

		self.entity:EndSerialization()
	end
end

function EntityScript:NetDeserialize()
	-- Save the components that need saving:
	if self.entity:HasNewDeserializationData() then

		self.entity:StartDeserialization()

		local ownerID = self.entity:DeserializeBoolean()
		self.owner = nil
		if ownerID then
			local entGUID = TheNet:FindGUIDForEntityID(self.entity:DeserializeEntityID())
			local ent = Ents[entGUID]
			if ent and ent:IsValid() then
				self.owner = ent
			end
		end

		-- Deserialize the stategraph if it exists:
		if self.sg then
			self.sg:OnNetDeserialize();
		end

		if self.boss_coro and self.boss_coro.OnNetDeserialize then
			self.boss_coro:OnNetDeserialize()
		end

		local nr = self.entity:DeserializeUInt(MaxNrComponentsBits)

		local lastComponentDeserialized = nil
		for i = 1, nr do
			-- save the hash of the name of the netcomponent:
			self.entity:AlignToByte();
			local hash = self.entity:DeserializeUInt(32)		-- 32 bit hash

			assert(hash ~= nil, "Error: " .. self.entity:GetPrefabName() .. " could not load component hash. Last component loaded:" .. (lastComponentDeserialized or "none"))

			if hash == nil then
				break;	-- Stop reading any more component data
			end

			-- find the component with this hash, or add it if it doesn't exist
			assert(self.netcomponents)
			local cmp = self.netcomponents[hash]

			-- Add the component if it is missing:
			lastComponentDeserialized = GetComponentNameForHash(hash)	-- Keep track of the last component that was serialized, so that we can see what component likely caused the read error if an error occurs later on
			if not cmp and lastComponentDeserialized then
				cmp = self:AddComponent(lastComponentDeserialized)
			end

			assert(cmp, "Error: " .. self.entity:GetPrefabName() .. "could not find or create component with hash "..hash .. " (".. (lastComponentDeserialized or "none") .. ")")
			if not cmp then
				break;	-- Stop reading any more component data
			end

			cmp:OnNetDeserialize()
		end

		self.entity:EndDeserialization()
	end

	self:ResolveInLimboTag();
end


function EntityScript:ResolveInLimboTag()
	-- Also update the in limbo tag:
	if self:IsInLimbo() ~= self._netinlimbo then
		self._netinlimbo = self:IsInLimbo()
		if self:IsInLimbo() then
			-- Entity should be in limbo, so hide the entity:
			self.entity:Hide()
			if self.brain ~= nil then
				self.brain:Pause("limbo")
			end
			if self.sg ~= nil then
				self.sg:Pause("limbo")
			end
			if self.Physics ~= nil then
				self.Physics:SetEnabled(false)
			end
			if self.Light ~= nil and self.Light:GetDisableOnSceneRemoval() then
				self.Light:Enable(false)
			end
			if self.SoundEmitter ~= nil then
				-- currently no use-cases of returning to scene with the
				-- expectation that autostop sounds resume.  If so, we'll need
				-- to pause and resume sounds instead of killing them
				self.SoundEmitter:KillAutoStopSounds()
			end
			if not self.paused and self.AnimState ~= nil then
				self.AnimState:Pause()
			end
			self:PushEvent("enterlimbo")
		else
			-- Entity should NOT be in limbo so make the entity show up:
			self.entity:Show()
			if self.Physics ~= nil then
				self.Physics:SetEnabled(true)
			end
			if self.Light ~= nil then
				self.Light:Enable(true)
			end
			if not self.paused and self.AnimState ~= nil then
				self.AnimState:Resume()
			end
			if self.brain ~= nil then
				self.brain:Resume("limbo")
			end
			if self.sg ~= nil then
				self.sg:Resume("limbo")
			end
			self:PushEvent("exitlimbo")
		end
	end
end
